<?php
session_start();
include 'db.php';

// ✅ تقييد الصفحة على الأدمن فقط
if (!isset($_SESSION['admin_id'])) {
    header("Location: admin_login.php");
    exit();
}

mysqli_set_charset($conn, "utf8mb4");

// دالة لتأمين المخرجات
function esc($s) {
    return htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
}

// دالة لطباعة الدرجة بشكل مرتب أو -
function fmt_grade($v) {
    if ($v === null) return '-';
    if ($v === '' || $v == 0) return '-';
    $num = number_format((float)$v, 2, '.', '');
    $num = rtrim(rtrim($num, '0'), '.');
    return $num;
}

// ✅ دالة لإرجاع كلاس اللون حسب الدرجة
function grade_class($v) {
    if ($v === null || $v === '' || $v == 0) return '';
    $g = floatval($v);
    if ($g < 50) {
        return 'grade-red';
    } elseif ($g < 60) {
        return 'grade-yellow';
    } elseif ($g < 80) {
        return 'grade-blue';
    } else {
        return 'grade-green';
    }
}

// ✅ دالة لأخذ أعلى 3 درجات
function top3_avg_admin($arr) {
    $clean = array();
    foreach ($arr as $v) {
        $clean[] = floatval($v);
    }
    rsort($clean);
    $top3 = array_slice($clean, 0, 3);
    $sum = array_sum($top3);
    $count = count($top3) > 0 ? count($top3) : 1;
    return round($sum / $count, 2);
}

// ✅ دالة تقريب نصف فأعلى
function round_half_up_int($v) {
    $v = (float)$v;
    $i = floor($v);
    return ($v - $i >= 0.5) ? $i + 1 : $i;
}

/* =========================
   1) جلب الفلاتر من GET
   ========================= */
$filter_stage = isset($_GET['stage']) ? trim($_GET['stage']) : '';
$filter_class = isset($_GET['class']) ? trim($_GET['class']) : '';
$search_name  = isset($_GET['q']) ? trim($_GET['q']) : '';

/* =========================
   2) جلب قائمة المراحل
   ========================= */
$stages = array();
$stg_q = mysqli_query($conn, "SELECT DISTINCT stage FROM students ORDER BY stage");
if ($stg_q) {
    while ($row = mysqli_fetch_assoc($stg_q)) {
        $stages[] = $row['stage'];
    }
}

/* =========================
   3) جلب الطلبة حسب الفلاتر
   ========================= */
$where = " WHERE 1 ";
if ($filter_stage != '') {
    $stage_esc = mysqli_real_escape_string($conn, $filter_stage);
    $where .= " AND s.stage = '".$stage_esc."'";
}
if ($filter_class != '') {
    $class_esc = mysqli_real_escape_string($conn, $filter_class);
    $where .= " AND s.class = '".$class_esc."'";
}
if ($search_name != '') {
    $q_esc = mysqli_real_escape_string($conn, $search_name);
    $where .= " AND s.name LIKE '%".$q_esc."%'";
}

$sql_students = "SELECT s.* 
                 FROM students s 
                 $where
                 ORDER BY s.stage, s.class, s.name";

$students_q = mysqli_query($conn, $sql_students);
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
<meta charset="UTF-8">
<title>كروت تقرير الطلبة | ثانوية الأوائل الأهلية للبنين</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;600;700&display=swap" rel="stylesheet">
<style>
*{box-sizing:border-box;margin:0;padding:0}
body{
    font-family:'AL-BATTAR','Calibri','Tajawal',sans-serif;
    background:radial-gradient(circle at 0% 0%,#1b2540,#050814 70%);
    color:#fff;
    min-height:100vh;
    padding:20px;
}
h1{
    font-size:26px;
    margin-bottom:5px;
}
.subtitle{
    color:#a7b3d9;
    margin-bottom:20px;
}
.container{
    max-width:1200px;
    margin:0 auto;
}

/* ✅ هيدر الصفحة + زر الرجوع */
header{
    display:flex;
    justify-content:space-between;
    align-items:center;
    margin-bottom:15px;
}
header h1{
    font-size:22px;
    margin-bottom:0;
}
header a{
    text-decoration:none;
    padding:7px 16px;
    border-radius:999px;
    background:linear-gradient(120deg,#00e6ff,#007bff);
    color:#fff;
    font-weight:600;
}

/* شريط الفلاتر */
.filters{
    background:rgba(15,23,42,0.9);
    border-radius:16px;
    padding:15px 20px;
    margin-bottom:20px;
    box-shadow:0 10px 30px rgba(0,0,0,0.35);
    display:flex;
    flex-wrap:wrap;
    gap:10px 15px;
    align-items:flex-end;
}
.filters .field{
    display:flex;
    flex-direction:column;
    font-size:13px;
}
.filters label{
    margin-bottom:3px;
    color:#cbd5f5;
}
.filters select,
.filters input{
    border-radius:10px;
    border:1px solid #323b5a;
    background:#020617;
    color:#e5e7eb;
    padding:6px 10px;
    min-width:130px;
}
.filters button{
    border:none;
    border-radius:999px;
    padding:7px 18px;
    background:linear-gradient(135deg,#3b82f6,#22c55e);
    color:#fff;
    font-weight:600;
    cursor:pointer;
    box-shadow:0 8px 20px rgba(34,197,94,0.4);
    transition:transform 0.1s ease,box-shadow 0.1s ease;
}
.filters button:hover{
    transform:translateY(-1px);
    box-shadow:0 10px 24px rgba(34,197,94,0.6);
}

/* كارت الطالب (عرض الشاشة) */
.student-card{
    background:rgba(15,23,42,0.95);
    border-radius:20px;
    padding:15px 18px 18px;
    margin-bottom:20px;
    box-shadow:0 12px 35px rgba(0,0,0,0.5);
    border:1px solid rgba(148,163,184,0.3);
    position:relative;
    overflow:hidden;
    page-break-inside:avoid;
}
.student-card::before{
    content:"";
    position:absolute;
    inset:-50%;
    background:radial-gradient(circle at 0 0,rgba(56,189,248,0.12),transparent 50%),
               radial-gradient(circle at 100% 100%,rgba(52,211,153,0.12),transparent 55%);
    opacity:0.7;
    pointer-events:none;
}
.student-card-inner{
    position:relative;
    z-index:2;
}

/* هيدر الكارت مع الشعار */
.card-header{
    display:flex;
    justify-content:space-between;
    align-items:center;
    margin-bottom:10px;
    gap:10px;
}
.card-logo{
    text-align:right;
}
.card-logo img.school-logo{
    max-height:55px;
    display:block;
}
.card-header-center{
    text-align:center;
    flex:1;
}
.school-name{
    font-weight:700;
    font-size:18px;
    font-family:'AL-BATTAR','Tajawal',sans-serif;
    letter-spacing:0.5px;
}
.card-title{
    font-size:13px;
    color:#a5b4fc;
}
.card-header-meta{
    text-align:left;
    font-size:11px;
    color:#9ca3af;
}

/* بيانات الطالب الأساسية */
.card-meta{
    display:flex;
    flex-wrap:wrap;
    gap:8px 16px;
    font-size:13px;
    margin-bottom:10px;
}
.meta-line span{
    color:#9ca3af;
}
.meta-line strong{
    color:#e5e7eb;
    font-weight:700;
}
.meta-name strong{
    font-size:17px;
}
.meta-stage strong,
.meta-class strong{
    font-size:15px;
}

/* جدول الدرجات */
.grades-box{
    margin-bottom:12px;
}
.grades-table{
    width:100%;
    border-collapse:collapse;
    font-size:11px;
    background:rgba(15,23,42,0.9);
}
.grades-table th,
.grades-table td{
    border:1px solid rgba(55,65,81,0.8);
    padding:4px 6px;
    text-align:center;
}
.grades-table th{
    background:rgba(15,23,42,0.95);
    font-weight:600;
    color:#e5e7eb;
}
.grades-table td.subject-col{
    text-align:right;
    font-weight:600;
}

/* تمييز صفوف المواد */
.grades-table tbody tr:nth-child(odd){
    background:rgba(15,23,42,0.95);
}
.grades-table tbody tr:nth-child(even){
    background:rgba(17,24,39,0.95);
}

/* تلوين الدرجات */
td.grade-cell{
    font-weight:600;
}
.grade-red{
    background:#c0392b !important;
    color:#fff !important;
}
.grade-yellow{
    background:#f1c40f !important;
    color:#000 !important;
}
.grade-blue{
    background:#2980b9 !important;
    color:#fff !important;
}
.grade-green{
    background:#27ae60 !important;
    color:#fff !important;
}

/* شطب الدرجة الأقل */
.strike-lowest{
    position:relative;
    font-weight:700;
}
.strike-lowest::after{
    content:"";
    position:absolute;
    left:3px;
    right:3px;
    top:50%;
    transform:translateY(-50%);
    border-top:1px solid #000;
    pointer-events:none;
}

/* أقسام التقييم والسلوك (شاشة فقط) */
.sections-row{
    display:flex;
    flex-wrap:wrap;
    gap:10px;
    margin-top:10px;
}
.section-box{
    flex:1 1 250px;
    background:rgba(15,23,42,0.9);
    border-radius:12px;
    padding:8px 10px;
    border:1px solid rgba(55,65,81,0.9);
}
.section-box h4{
    font-size:13px;
    margin-bottom:6px;
}
.section-box .small{
    font-size:11px;
    color:#9ca3af;
    margin-bottom:4px;
}
.section-box ul{
    list-style:none;
    font-size:11px;
    max-height:160px;
    overflow:auto;
}
.section-box li{
    border-bottom:1px dashed rgba(75,85,99,0.8);
    padding:4px 0;
}
.section-box li:last-child{
    border-bottom:none;
}
.tag{
    display:inline-block;
    border-radius:999px;
    padding:1px 7px;
    font-size:10px;
    margin-left:4px;
}
.tag.beh-good{background:rgba(16,185,129,0.15);color:#6ee7b7;border:1px solid rgba(16,185,129,0.5);}
.tag.beh-weak{background:rgba(248,113,113,0.15);color:#fecaca;border:1px solid rgba(248,113,113,0.5);}
.tag.level-high{background:rgba(59,130,246,0.15);color:#bfdbfe;border:1px solid rgba(59,130,246,0.5);}

/* منطقة التواقيع */
.signatures-row{
    display:flex;
    justify-content:space-between;
    align-items:flex-end;
    gap:10px;
    margin-top:26px;
}
.sig-col{
    flex:1 1 0;
    font-size:12px;
}
.sig-parent{
    text-align:center;
}
.sig-director{
    text-align:center;
}
.sig-label{
    margin-bottom:6px;
    font-weight:600;
}
.sig-line{
    padding-top:6px;
    min-height:26px;
    border-top:none;
}
.director-name{
    font-weight:700;
    font-size:13px;
    letter-spacing:0.5px;
    padding-top:6px;
}

/* أزرار الطباعة */
.card-footer{
    margin-top:10px;
    text-align:left;
}
.card-footer button{
    border:none;
    border-radius:999px;
    padding:5px 14px;
    font-size:12px;
    background:linear-gradient(135deg,#38bdf8,#6366f1);
    color:#fff;
    cursor:pointer;
    box-shadow:0 6px 15px rgba(59,130,246,0.5);
}
.card-footer button:hover{
    transform:translateY(-1px);
}

/* حالة لا توجد بيانات */
.empty{
    margin-top:30px;
    text-align:center;
    color:#9ca3af;
}

/* طباعة */
.no-print{ }

@media print{
    body{
        font-family:'AL-BATTAR','Calibri','Tajawal',sans-serif;
        background:
          radial-gradient(circle at 10% 0%, #f9fafb 0%, #eef2ff 35%, #e0f2fe 65%, #f9fafb 100%);
        color:#000;
        padding:0;
        margin:0;
        -webkit-print-color-adjust:exact;
        print-color-adjust:exact;
    }
    .no-print{
        display:none !important;
    }

    .student-card{
        background:linear-gradient(145deg,#ffffff 0%,#f8fafc 55%,#eef2ff 100%) !important;
        color:#000 !important;
        box-shadow:none !important;
        border:4px double #000;
        border-radius:18px;
        margin:18px;
        padding:24px 26px;
        page-break-after:always;
        position:relative;
        overflow:visible;
    }
    .student-card::before{
        content:"";
        position:absolute;
        inset:7px;
        border:1px dashed #000;
        border-radius:14px;
        pointer-events:none;
        background:none !important;
        opacity:1 !important;
    }
    .student-card::after{
        content:"◆ ◆ ◆";
        position:absolute;
        top:-10px;
        left:50%;
        transform:translateX(-50%);
        font-size:10px;
        font-weight:bold;
    }

    /* ✅ إظهار شعار الأوائل على اليمين في الطباعة */
    .card-header{
        display:flex;
        justify-content:space-between;
        align-items:flex-start;
        gap:10px;
    }
    .card-logo{
        text-align:right;
    }
    .card-logo img.school-logo{
        max-height:60px;
    }

    .grades-table th,
    .grades-table td{
        border-color:#000;
    }

    /* ✅ لون نص المواد أبيض في الطباعة (عنوان "المادة" واسم المادة) */
    .grades-table th:first-child,
    .grades-table td.subject-col{
        color:#fff !important;
    }

    .grade-red{
        background:#c0392b !important;
        color:#fff !important;
    }
    .grade-yellow{
        background:#f1c40f !important;
        color:#000 !important;
    }
    .grade-blue{
        background:#2980b9 !important;
        color:#fff !important;
    }
    .grade-green{
        background:#27ae60 !important;
        color:#fff !important;
    }

    .strike-lowest::after{
        border-top:1px solid #000;
    }

    /* ✅ ثانوية الأوائل الأهلية للبنين في الوسط وباللون الأسود */
    .school-name{
        font-family:'AL-BATTAR','Tajawal',sans-serif;
        font-size:22px;
        letter-spacing:1px;
        text-align:center;
        color:#000;
        font-weight:700;
    }
    .card-title{
        font-size:13px;
    }

    /* ✅ اسم الطالب + المرحلة + الشعبة: Bold + أسود */
    .meta-line span,
    .meta-line strong{
        color:#000 !important;
        font-weight:700;
    }
    .meta-name strong{
        font-size:19px;
    }
    .meta-stage strong,
    .meta-class strong{
        font-size:17px;
    }

    .card-header-meta{
        font-size:12px;
    }

    .signatures-row{
        display:flex;
        justify-content:space-between;
        align-items:flex-end;
        gap:10px;
        margin-top:30px;
        font-size:12px;
    }
    .sig-parent{
        text-align:center;
    }
    .sig-director{
        text-align:center;
    }

    .sections-row{
        display:none !important;
    }
}
</style>
</head>
<body>

<!-- ✅ هيدر الصفحة مع زر الرجوع -->
<header class="no-print">
    <h1>كروت تقرير الطلبة</h1>
    <a href="admin_dashboard.php">↩ العودة للوحة التحكم</a>
</header>

<div class="container">
    <div class="subtitle">عرض درجات الطالب (ش1–ش6، الفصول، نصف السنة، النتيجة السنوية) في كارت واحد.</div>

    <!-- ✅ شريط الفلاتر + زر طباعة جميع الكروت -->
    <form method="get" class="filters no-print">
        <div class="field">
            <label for="stage">المرحلة</label>
            <select name="stage" id="stage">
                <option value="">جميع المراحل</option>
                <?php
                foreach ($stages as $st) {
                    $sel = ($st == $filter_stage) ? 'selected' : '';
                    echo '<option value="'.esc($st).'" '.$sel.'>'.esc($st).'</option>';
                }
                ?>
            </select>
        </div>

        <div class="field">
            <label for="class">الشعبة</label>
            <select name="class" id="class">
                <option value="">جميع الشعب</option>
                <?php
                $classes = array('A','B','C','D','E','F');
                foreach ($classes as $c) {
                    $sel = ($c == $filter_class) ? 'selected' : '';
                    echo '<option value="'.esc($c).'" '.$sel.'>'.$c.'</option>';
                }
                ?>
            </select>
        </div>

        <div class="field">
            <label for="q">بحث باسم الطالب</label>
            <input type="text" name="q" id="q" placeholder="اكتب جزء من الاسم" value="<?php echo esc($search_name); ?>">
        </div>

        <div class="field">
            <label for="period">الفترة المطلوبة</label>
            <select id="period" onchange="updateColumns();">
                <option value="all">كل الدرجات</option>
                <option value="f1">الفصل الأول (ش1+ش2+ش3+ف1)</option>
                <option value="mid">نصف السنة</option>
                <option value="f2">الفصل الثاني (ش4+ش5+ش6+ف2)</option>
                <option value="annual">النتيجة السنوية</option>
            </select>
        </div>

        <div class="field">
            <label>&nbsp;</label>
            <button type="submit">تطبيق الفلاتر</button>
        </div>

        <!-- ✅ زر طباعة جميع الكروت كـ PDF (من المتصفح) -->
        <div class="field">
            <label>&nbsp;</label>
            <button type="button" onclick="printAllCards()">🖨️ طباعة جميع الكروت (PDF)</button>
        </div>
    </form>

    <?php
    if (!$students_q || mysqli_num_rows($students_q) == 0) {
        echo '<div class="empty">لا توجد طلبة مطابقة للفلاتر الحالية.</div>';
    } else {

        while ($stu = mysqli_fetch_assoc($students_q)) {

            $stu_id = (int)$stu['id'];

            // 4) جلب درجات الطالب
            $grades_map = array();
            $g_sql = "SELECT * FROM grades 
                      WHERE student_id = ".$stu_id."
                      ORDER BY subject, saved_at DESC, id DESC";
            $g_q = mysqli_query($conn, $g_sql);
            if ($g_q) {
                while ($g = mysqli_fetch_assoc($g_q)) {
                    $subj = isset($g['subject']) && $g['subject'] != '' ? $g['subject'] : 'غير محدد';
                    if (!isset($grades_map[$subj])) {
                        $grades_map[$subj] = $g;
                    }
                }
            }

            // ترتيب المواد
            if (count($grades_map) > 0) {
                $desired_order = array(
                    'التربية الاسلامية',
                    'اللغة العربية',
                    'اللغة الانكليزية',
                    'الرياضيات',
                    'الكيمياء',
                    'الاحياء',
                    'الفيزياء',
                    'الاجتماعيات',
                    'الحاسوب',
                    'اللغة الفرنسية',
                    'اللغة الكردية'
                );
                $ordered_grades = array();

                foreach ($desired_order as $subjName) {
                    if (isset($grades_map[$subjName])) {
                        $ordered_grades[$subjName] = $grades_map[$subjName];
                    }
                }
                foreach ($grades_map as $subj => $g) {
                    if (!isset($ordered_grades[$subj])) {
                        $ordered_grades[$subj] = $g;
                    }
                }
            } else {
                $ordered_grades = array();
            }

            // 5) تقييمات الطالب
            $evals = array();
            $e_sql = "SELECT e.*, t.name AS teacher_name
                      FROM evaluations e
                      LEFT JOIN teachers t ON e.teacher_id = t.id
                      WHERE e.student_id = ".$stu_id."
                      ORDER BY e.created_at DESC";
            $e_q = mysqli_query($conn, $e_sql);
            if ($e_q) {
                while ($e = mysqli_fetch_assoc($e_q)) {
                    $evals[] = $e;
                }
            }

            // 6) ملاحظات السلوك من الإدارة
            $behaviors = array();
            $b_sql = "SELECT * FROM admin_behavior 
                      WHERE student_id = ".$stu_id."
                      ORDER BY problem_date DESC, id DESC";
            $b_q = mysqli_query($conn, $b_sql);
            if ($b_q) {
                while ($b = mysqli_fetch_assoc($b_q)) {
                    $behaviors[] = $b;
                }
            }
            ?>

            <div class="student-card" id="card-<?php echo $stu_id; ?>">
                <div class="student-card-inner">
                    <div class="card-header">
                        <div class="card-logo">
                            <img src="bookk.png" alt="شعار المدرسة" class="school-logo">
                        </div>
                        <div class="card-header-center">
                            <div class="school-name">ثانوية الأوائل الأهلية للبنين</div>
                            <div class="card-title">كارت تقرير الطالب الشامل</div>
                        </div>
                        <div class="card-header-meta">
                            التاريخ: <?php echo date('d-m-Y'); ?><br>
                            رقم الطالب: <?php echo $stu_id; ?>
                        </div>
                    </div>

                    <div class="card-meta">
                        <div class="meta-line meta-name"><span>اسم الطالب:</span> <strong><?php echo esc($stu['name']); ?></strong></div>
                        <div class="meta-line meta-stage"><span>المرحلة:</span> <strong><?php echo esc($stu['stage']); ?></strong></div>
                        <div class="meta-line meta-class"><span>الشعبة:</span> <strong><?php echo esc($stu['class']); ?></strong></div>
                    </div>

                    <!-- جدول الدرجات -->
                    <div class="grades-box">
                        <table class="grades-table">
                            <thead>
                            <tr>
                                <th>المادة</th>
                                <th class="col-m1">ش1</th>
                                <th class="col-m2">ش2</th>
                                <th class="col-m3">ش3</th>
                                <th class="col-d1">يومي 1</th>
                                <th class="col-sem1">فصل أول</th>
                                <th class="col-mid">نصف السنة</th>
                                <th class="col-m4">ش4</th>
                                <th class="col-m5">ش5</th>
                                <th class="col-m6">ش6</th>
                                <th class="col-d2">يومي 2</th>
                                <th class="col-sem2">فصل ثاني</th>
                                <th class="col-avg">المعدل السنوي</th>
                                <th class="col-final">النتيجة النهائية</th>
                            </tr>
                            </thead>
                            <tbody>
                            <?php
                            if (count($ordered_grades) == 0) {
                                echo '<tr><td colspan="14">لا توجد درجات مسجلة لهذا الطالب حتى الآن.</td></tr>';
                            } else {
                                foreach ($ordered_grades as $subj => $g) {

                                    $m1  = isset($g['month1'])  ? (float)$g['month1']  : 0;
                                    $m2  = isset($g['month2'])  ? (float)$g['month2']  : 0;
                                    $m3  = isset($g['month3'])  ? (float)$g['month3']  : 0;
                                    $d1  = isset($g['daily1'])  ? (float)$g['daily1']  : 0;
                                    $mid = isset($g['midterm']) ? (float)$g['midterm'] : 0;
                                    $m4  = isset($g['month4'])  ? (float)$g['month4']  : 0;
                                    $m5  = isset($g['month5'])  ? (float)$g['month5']  : 0;
                                    $m6  = isset($g['month6'])  ? (float)$g['month6']  : 0;
                                    $d2  = isset($g['daily2'])  ? (float)$g['daily2']  : 0;

                                    if ($d1 > 0) {
                                        $firstVals = array(
                                            'month1' => $m1,
                                            'month2' => $m2,
                                            'month3' => $m3,
                                            'daily1' => $d1
                                        );
                                        $min_key1 = null;
                                        $min_val1 = null;
                                        foreach ($firstVals as $k => $v) {
                                            if ($min_val1 === null || $v < $min_val1) {
                                                $min_val1 = $v;
                                                $min_key1 = $k;
                                            }
                                        }
                                        $sem1_calc = top3_avg_admin(array($m1, $m2, $m3, $d1));
                                    } else {
                                        $min_key1 = null;
                                        $sem1_calc = round(($m1 + $m2 + $m3) / 3, 2);
                                    }
                                    $sem1_calc = round_half_up_int($sem1_calc);

                                    if ($d2 > 0) {
                                        $secondVals = array(
                                            'month4' => $m4,
                                            'month5' => $m5,
                                            'month6' => $m6,
                                            'daily2' => $d2
                                        );
                                        $min_key2 = null;
                                        $min_val2 = null;
                                        foreach ($secondVals as $k => $v) {
                                            if ($min_val2 === null || $v < $min_val2) {
                                                $min_val2 = $v;
                                                $min_key2 = $k;
                                            }
                                        }
                                        $sem2_calc = top3_avg_admin(array($m4, $m5, $m6, $d2));
                                    } else {
                                        $min_key2 = null;
                                        $sem2_calc = round(($m4 + $m5 + $m6) / 3, 2);
                                    }
                                    $sem2_calc = round_half_up_int($sem2_calc);

                                    $annual_calc = round(($sem1_calc + $sem2_calc + $mid) / 3, 2);

                                    $cls_m1   = grade_class($g['month1']);
                                    $cls_m2   = grade_class($g['month2']);
                                    $cls_m3   = grade_class($g['month3']);
                                    $cls_d1   = grade_class($g['daily1']);
                                    $cls_sem1 = grade_class($sem1_calc);
                                    $cls_mid  = grade_class($g['midterm']);
                                    $cls_m4   = grade_class($g['month4']);
                                    $cls_m5   = grade_class($g['month5']);
                                    $cls_m6   = grade_class($g['month6']);
                                    $cls_d2   = grade_class($g['daily2']);
                                    $cls_sem2 = grade_class($sem2_calc);
                                    $cls_avg  = grade_class($annual_calc);
                                    $cls_fin  = grade_class($g['final_result']);

                                    echo '<tr>';
                                    echo '<td class="subject-col">'.esc($subj).'</td>';
                                    echo '<td class="col-m1 grade-cell '.$cls_m1.(($min_key1 === "month1") ? ' strike-lowest' : '').'">'.fmt_grade($g['month1']).'</td>';
                                    echo '<td class="col-m2 grade-cell '.$cls_m2.(($min_key1 === "month2") ? ' strike-lowest' : '').'">'.fmt_grade($g['month2']).'</td>';
                                    echo '<td class="col-m3 grade-cell '.$cls_m3.(($min_key1 === "month3") ? ' strike-lowest' : '').'">'.fmt_grade($g['month3']).'</td>';
                                    echo '<td class="col-d1 grade-cell '.$cls_d1.(($min_key1 === "daily1") ? ' strike-lowest' : '').'">'.fmt_grade($g['daily1']).'</td>';
                                    echo '<td class="col-sem1 grade-cell '.$cls_sem1.'">'.fmt_grade($sem1_calc).'</td>';
                                    echo '<td class="col-mid grade-cell '.$cls_mid.'">'.fmt_grade($g['midterm']).'</td>';
                                    echo '<td class="col-m4 grade-cell '.$cls_m4.(($min_key2 === "month4") ? ' strike-lowest' : '').'">'.fmt_grade($g['month4']).'</td>';
                                    echo '<td class="col-m5 grade-cell '.$cls_m5.(($min_key2 === "month5") ? ' strike-lowest' : '').'">'.fmt_grade($g['month5']).'</td>';
                                    echo '<td class="col-m6 grade-cell '.$cls_m6.(($min_key2 === "month6") ? ' strike-lowest' : '').'">'.fmt_grade($g['month6']).'</td>';
                                    echo '<td class="col-d2 grade-cell '.$cls_d2.(($min_key2 === "daily2") ? ' strike-lowest' : '').'">'.fmt_grade($g['daily2']).'</td>';
                                    echo '<td class="col-sem2 grade-cell '.$cls_sem2.'">'.fmt_grade($sem2_calc).'</td>';
                                    echo '<td class="col-avg grade-cell '.$cls_avg.'">'.fmt_grade($annual_calc).'</td>';
                                    echo '<td class="col-final grade-cell '.$cls_fin.'">'.fmt_grade($g['final_result']).'</td>';
                                    echo '</tr>';
                                }
                            }
                            ?>
                            </tbody>
                        </table>
                    </div>

                    <!-- التقييم والسلوك (شاشة فقط) -->
                    <div class="sections-row no-print">
                        <div class="section-box">
                            <h4>تقييمات المدرسين (سلوك + مستوى + ملاحظات)</h4>
                            <div class="small">يتم ملؤها من صفحة تقييم الطلبة.</div>
                            <?php if (count($evals) == 0): ?>
                                <div class="small">لا توجد تقييمات مسجلة لهذا الطالب.</div>
                            <?php else: ?>
                                <ul>
                                    <?php foreach ($evals as $e): ?>
                                        <li>
                                            <strong><?php echo esc($e['subject'] != '' ? $e['subject'] : 'بدون مادة'); ?></strong>
                                            <?php
                                            $beh = isset($e['behavior']) ? trim($e['behavior']) : '';
                                            $lvl = isset($e['academic_level']) ? trim($e['academic_level']) : '';
                                            $b_class = '';
                                            if ($beh != '') {
                                                if (mb_strpos($beh,'ضعيف') !== false) $b_class = 'beh-weak';
                                                elseif (mb_strpos($beh,'ممتاز') !== false || mb_strpos($beh,'جيد') !== false) $b_class = 'beh-good';
                                            }
                                            $l_class = '';
                                            if ($lvl != '') {
                                                if (mb_strpos($lvl,'ممتاز') !== false || mb_strpos($lvl,'جيد') !== false) $l_class = 'level-high';
                                                elseif (mb_strpos($lvl,'ضعيف') !== false) $l_class = 'beh-weak';
                                            }
                                            ?>
                                            <?php if ($beh != ''): ?>
                                                <span class="tag <?php echo $b_class; ?>">سلوك: <?php echo esc($beh); ?></span>
                                            <?php endif; ?>
                                            <?php if ($lvl != ''): ?>
                                                <span class="tag <?php echo $l_class; ?>">مستوى: <?php echo esc($lvl); ?></span>
                                            <?php endif; ?>
                                            <?php if ($e['teacher_name'] != ''): ?>
                                                <div>المعلم: <?php echo esc($e['teacher_name']); ?></div>
                                            <?php endif; ?>
                                            <?php if ($e['notes'] != ''): ?>
                                                <div>ملاحظات: <?php echo nl2br(esc($e['notes'])); ?></div>
                                            <?php endif; ?>
                                        </li>
                                    <?php endforeach; ?>
                                </ul>
                            <?php endif; ?>
                        </div>

                        <div class="section-box">
                            <h4>ملاحظات سلوكية من الإدارة</h4>
                            <div class="small">من سجل المخالفات السلوكية في جدول admin_behavior.</div>
                            <?php if (count($behaviors) == 0): ?>
                                <div class="small">لا توجد ملاحظات سلوكية مسجلة من الإدارة.</div>
                            <?php else: ?>
                                <ul>
                                    <?php foreach ($behaviors as $b): ?>
                                        <li>
                                            <strong><?php echo esc($b['problem_date']); ?></strong> —
                                            <?php echo esc($b['problem_type']); ?>
                                            <?php if ($b['problem_count'] > 1): ?>
                                                <span class="tag beh-weak">تكرار: <?php echo (int)$b['problem_count']; ?></span>
                                            <?php endif; ?>
                                            <?php if ($b['problem_note'] != ''): ?>
                                                <div><?php echo nl2br(esc($b['problem_note'])); ?></div>
                                            <?php endif; ?>
                                            <div class="small">سجلت بواسطة: <?php echo esc($b['admin_name']); ?></div>
                                        </li>
                                    <?php endforeach; ?>
                                </ul>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- التواقيع -->
                    <div class="signatures-row">
                        <div class="sig-col sig-parent">
                            <div class="sig-label">توقيع ولي الأمر</div>
                            <div class="sig-line"></div>
                        </div>
                        <div class="sig-col sig-director">
                            <div class="sig-label">توقيع المدير</div>
                            <div class="director-name">أ. يوسف هيثم ابراهيم</div>
                        </div>
                    </div>

                    <div class="card-footer no-print">
                        <button type="button" onclick="printStudentCard(<?php echo $stu_id; ?>)">🖨️ طباعة هذا الكارت</button>
                    </div>
                </div>
            </div>

            <?php
        } // end while students
    }
    ?>
</div>

<script>
// ✅ إخفاء/إظهار الأعمدة حسب الفترة
function updateColumns(){
    var period = document.getElementById('period').value;
    var allClasses = [
        'col-m1','col-m2','col-m3',
        'col-sem1','col-mid',
        'col-m4','col-m5','col-m6',
        'col-sem2',
        'col-avg','col-final',
        'col-d1','col-d2'
    ];

    for (var i=0;i<allClasses.length;i++){
        var cls = allClasses[i];
        var cells = document.getElementsByClassName(cls);
        for (var j=0;j<cells.length;j++){
            cells[j].style.display = '';
        }
    }

    function hide(list){
        for (var i=0;i<list.length;i++){
            var cls = list[i];
            var cells = document.getElementsByClassName(cls);
            for (var j=0;j<cells.length;j++){
                cells[j].style.display = 'none';
            }
        }
    }

    if (period === 'f1'){
        hide(['col-mid','col-m4','col-m5','col-m6','col-sem2','col-avg','col-final','col-d2']);
    } else if (period === 'mid'){
        hide(['col-m1','col-m2','col-m3','col-sem1','col-m4','col-m5','col-m6','col-sem2','col-avg','col-final','col-d1','col-d2']);
    } else if (period === 'f2'){
        hide(['col-m1','col-m2','col-m3','col-sem1','col-mid','col-avg','col-final','col-d1']);
    } else if (period === 'annual'){
        hide(['col-m1','col-m2','col-m3','col-sem1','col-mid','col-m4','col-m5','col-m6','col-sem2','col-d1','col-d2']);
    }
}

window.addEventListener('load', updateColumns);

// ✅ طباعة جميع الكروت الظاهرة كـ PDF (من المتصفح)
function printAllCards(){
    window.print();
}

// ✅ طباعة كارت طالب واحد فقط بنفس تنسيقات طباعة جميع الكروت
function printStudentCard(id){
    var card = document.getElementById('card-' + id);
    if (!card) return;

    var win = window.open('', '_blank', 'width=900,height=1200');
    win.document.write('<html dir="rtl" lang="ar"><head><title>كارت الطالب</title>');
    win.document.write('<meta charset="UTF-8">');
    win.document.write('<style>');
    // نفس تنسيقات @media print لكن بدون media (تُطبَّق مباشرة في نافذة الطباعة)
    win.document.write('body{font-family:AL-BATTAR, Calibri, Tajawal, sans-serif;direction:rtl;margin:0;padding:0;background:radial-gradient(circle at 10% 0%, #f9fafb 0%, #eef2ff 35%, #e0f2fe 65%, #f9fafb 100%);color:#000;-webkit-print-color-adjust:exact;print-color-adjust:exact;}');
    win.document.write('.student-card{background:linear-gradient(145deg,#ffffff 0%,#f8fafc 55%,#eef2ff 100%) !important;color:#000 !important;box-shadow:none !important;border:4px double #000;border-radius:18px;margin:18px;padding:24px 26px;page-break-after:always;position:relative;overflow:visible;}');
    win.document.write('.student-card::before{content:"";position:absolute;inset:7px;border:1px dashed #000;border-radius:14px;pointer-events:none;background:none !important;opacity:1 !important;}');
    win.document.write('.student-card::after{content:"◆ ◆ ◆";position:absolute;top:-10px;left:50%;transform:translateX(-50%);font-size:10px;font-weight:bold;}');
    win.document.write('.card-header{display:flex;justify-content:space-between;align-items:flex-start;gap:10px;}');
    win.document.write('.card-logo{text-align:right;}');
    win.document.write('.card-logo img.school-logo{max-height:60px;}');

    // ✅ نفس تنسيق الجدول في الطباعة العامة (خلفية لكل الأعمدة والصفوف)
    win.document.write('.grades-table{width:100%;border-collapse:collapse;font-size:11px;background:rgba(15,23,42,0.9);}');
    win.document.write('.grades-table th,.grades-table td{border:1px solid #000;padding:4px 6px;text-align:center;}');
    win.document.write('.grades-table th{background:rgba(15,23,42,0.95);font-weight:600;color:#e5e7eb;}');
    win.document.write('.grades-table tbody tr:nth-child(odd){background:rgba(15,23,42,0.95);}');
    win.document.write('.grades-table tbody tr:nth-child(even){background:rgba(17,24,39,0.95);}');
    win.document.write('.grades-table td.subject-col{text-align:right;font-weight:600;}');
    win.document.write('.grades-table th:first-child,.grades-table td.subject-col{color:#fff !important;}');

    win.document.write('.grade-cell{font-weight:600;}');
    win.document.write('.grade-red{background:#c0392b !important;color:#fff !important;}');
    win.document.write('.grade-yellow{background:#f1c40f !important;color:#000 !important;}');
    win.document.write('.grade-blue{background:#2980b9 !important;color:#fff !important;}');
    win.document.write('.grade-green{background:#27ae60 !important;color:#fff !important;}');

    win.document.write('.strike-lowest{position:relative;font-weight:700;}');
    win.document.write('.strike-lowest::after{content:"";position:absolute;left:3px;right:3px;top:50%;transform:translateY(-50%);border-top:1px solid #000;pointer-events:none;}');

    win.document.write('.card-meta{display:flex;flex-wrap:wrap;gap:8px 16px;font-size:13px;margin-bottom:10px;}');
    win.document.write('.meta-line span,.meta-line strong{color:#000 !important;font-weight:700;}');
    win.document.write('.meta-name strong{font-size:19px;}');
    win.document.write('.meta-stage strong,.meta-class strong{font-size:17px;}');
    win.document.write('.school-name{font-family:AL-BATTAR, Tajawal, sans-serif;font-size:22px;letter-spacing:1px;text-align:center;color:#000;font-weight:700;}');
    win.document.write('.card-title{font-size:13px;text-align:center;color:#555;}');
    win.document.write('.card-header-meta{font-size:12px;text-align:left;color:#000;}');
    win.document.write('.signatures-row{display:flex;justify-content:space-between;align-items:flex-end;gap:10px;margin-top:30px;font-size:12px;}');
    win.document.write('.sig-col{flex:1 1 0;}');
    win.document.write('.sig-parent{text-align:center;}');
    win.document.write('.sig-director{text-align:center;}');
    win.document.write('.sig-label{margin-bottom:6px;font-weight:600;}');
    win.document.write('.sig-line{padding-top:6px;min-height:26px;border-top:none;}');
    win.document.write('.director-name{font-weight:700;font-size:13px;letter-spacing:0.5px;padding-top:6px;}');
    win.document.write('.sections-row{display:none !important;}');
    win.document.write('.card-footer,.no-print{display:none !important;}');
    win.document.write('</style>');
    win.document.write('</head><body>');
    // نطبع الكارت كما هو مع الحواف والبوردَر
    win.document.write(card.outerHTML);
    win.document.write('</body></html>');
    win.document.close();
    win.focus();
    win.print();
}
</script>
</body>
</html>
