<?php
session_start();
include 'db.php';

// ✅ التحقق من تسجيل دخول الأدمن
if (!isset($_SESSION['admin_id'])) {
    header("Location: admin_login.php");
    exit();
}

mysqli_set_charset($conn, "utf8mb4");

// ✅ جلب بيانات الأدمن من الجلسة
$admin_username = isset($_SESSION['admin_username']) ? $_SESSION['admin_username'] : '';
$admin_role     = isset($_SESSION['admin_role']) ? $_SESSION['admin_role'] : null;

// ✅ لو ماكو role (جلسة قديمة) نحدده من اسم المستخدم
if ($admin_role === null) {
    switch ($admin_username) {
        case 'alawael0':
            // فقط كارت أسئلة المدرسين
            $admin_role = 'questions_only';
            break;

        case 'alawael1':
            // عرض الواجبات + المادة الداخلة بالامتحان + إحصائية الواجبات
            $admin_role = 'homework_exam_stats';
            break;

        case 'alawael2':
            // تقييمات + إحصائيات + درجات الطلبة للمواد + الواجبات + إحصائية الواجبات
            $admin_role = 'eval_stats_grades';
            break;

        case 'awael':   // اليوزر القديم
        case 'alawael': // اليوزر الكامل
        default:
            // كامل الصلاحيات
            $admin_role = 'super';
            break;
    }
    $_SESSION['admin_role'] = $admin_role;
}

// ✅ السماح فقط لـ super و eval_stats_grades بالدخول لهذه الصفحة
if ($admin_role !== 'super' && $admin_role !== 'eval_stats_grades') {
    header("Location: admin_dashboard.php");
    exit();
}

// دالة بسيطة لتأمين المخرجات
function esc($s) {
    return htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
}

/* =========================
   دوال حساب الفصلين والمعدل
   ========================= */

// ✅ دالة لأخذ أعلى 3 درجات (نفس منطق كارت التقرير)
function top3_avg_admin($arr) {
    $clean = array();
    foreach ($arr as $v) {
        $clean[] = floatval($v);
    }
    rsort($clean);
    $top3 = array_slice($clean, 0, 3);
    $sum  = array_sum($top3);
    $count = count($top3) > 0 ? count($top3) : 1;
    return round($sum / $count, 2);
}

// ✅ دالة تقريب نصف فأعلى
function round_half_up_int($v) {
    $v = (float)$v;
    $i = floor($v);
    return ($v - $i >= 0.5) ? $i + 1 : $i;
}

// ✅ حساب الفصل الأول بناءً على (ش1، ش2، ش3، يومي1) مع شطب الأقل منطقياً
function calc_semester1_from_row($g) {
    $m1 = isset($g['month1']) ? (float)$g['month1'] : 0;
    $m2 = isset($g['month2']) ? (float)$g['month2'] : 0;
    $m3 = isset($g['month3']) ? (float)$g['month3'] : 0;
    $d1 = isset($g['daily1']) ? (float)$g['daily1'] : 0;

    if ($d1 > 0) {
        // نحسب أعلى 3 درجات من الأربعة (الأقل يُهمَل)
        $sem1 = top3_avg_admin(array($m1, $m2, $m3, $d1));
    } else {
        // بدون اليومي: متوسط الشهور الثلاثة
        $sem1 = round(($m1 + $m2 + $m3) / 3, 2);
    }
    return round_half_up_int($sem1);
}

// ✅ حساب الفصل الثاني بناءً على (ش4، ش5، ش6، يومي2) مع شطب الأقل منطقياً
function calc_semester2_from_row($g) {
    $m4 = isset($g['month4']) ? (float)$g['month4'] : 0;
    $m5 = isset($g['month5']) ? (float)$g['month5'] : 0;
    $m6 = isset($g['month6']) ? (float)$g['month6'] : 0;
    $d2 = isset($g['daily2']) ? (float)$g['daily2'] : 0;

    if ($d2 > 0) {
        $sem2 = top3_avg_admin(array($m4, $m5, $m6, $d2));
    } else {
        $sem2 = round(($m4 + $m5 + $m6) / 3, 2);
    }
    return round_half_up_int($sem2);
}

// ✅ حساب المعدل السنوي (ف1 + ف2 + نصف السنة) ÷ 3 بنفس المنطق
function calc_annual_from_row($g) {
    $sem1 = calc_semester1_from_row($g);
    $sem2 = calc_semester2_from_row($g);
    $mid  = isset($g['midterm']) ? (float)$g['midterm'] : 0;

    $annual = round(($sem1 + $sem2 + $mid) / 3, 2);
    return round_half_up_int($annual);
}

// ✅ دالة عامة ترجع الدرجة حسب نوع الدرجة المطلوب
function get_grade_value_from_row($g, $grade_type) {
    if (!$g || !is_array($g)) {
        return null;
    }

    switch ($grade_type) {
        case 'semester1':
            return calc_semester1_from_row($g);
        case 'semester2':
            return calc_semester2_from_row($g);
        case 'annual_average':
            return calc_annual_from_row($g);
        default:
            if (isset($g[$grade_type]) && $g[$grade_type] !== '' && $g[$grade_type] !== null) {
                return floatval($g[$grade_type]);
            }
            return null;
    }
}

// ------------------------------
// ١) إعداد الفلاتر
// ------------------------------
$stage      = isset($_GET['stage']) ? trim($_GET['stage']) : '';
$grade_type = isset($_GET['grade_type']) ? trim($_GET['grade_type']) : 'month1';

// أنواع الدرجات المتاحة وربطها بأعمدة جدول grades
$grade_types = array(
    'month1'         => 'الشهر الأول',
    'month2'         => 'الشهر الثاني',
    'month3'         => 'الشهر الثالث',
    'semester1'      => 'الفصل الأول',
    'midterm'        => 'نصف السنة',
    'month4'         => 'الشهر الرابع',
    'month5'         => 'الشهر الخامس',
    'month6'         => 'الشهر السادس',
    'semester2'      => 'الفصل الثاني',
    'annual_average' => 'المعدل السنوي',
    'final_result'   => 'النتيجة النهائية'
);

// تأكيد أن نوع الدرجة صحيح
if (!isset($grade_types[$grade_type])) {
    $grade_type = 'month1';
}
$grade_col = $grade_type; // نستخدمه فقط للأنواع العادية، أما الفصلين والمعدل نحسبهم من الدوال

// ------------------------------
// ٢) تصدير Excel لجميع المراحل
// ------------------------------
if (isset($_GET['export']) && $_GET['export'] === 'excel_all') {

    $students_all   = array();
    $order_all      = array();
    $subjects_all   = array();

    $sql_all = "
        SELECT 
            s.id   AS student_id,
            s.name AS student_name,
            s.stage,
            s.class,
            g.subject,
            g.month1,
            g.month2,
            g.month3,
            g.daily1,
            g.semester1,
            g.midterm,
            g.month4,
            g.month5,
            g.month6,
            g.daily2,
            g.semester2,
            g.annual_average,
            g.final_result
        FROM students s
        LEFT JOIN grades g ON g.student_id = s.id
        ORDER BY s.stage, s.class, s.name
    ";

    $res_all = mysqli_query($conn, $sql_all);
    if ($res_all) {
        while ($row = mysqli_fetch_assoc($res_all)) {
            $sid = $row['student_id'];

            if (!isset($students_all[$sid])) {
                $students_all[$sid] = array(
                    'name'   => $row['student_name'],
                    'stage'  => $row['stage'],
                    'class'  => $row['class'],
                    'grades' => array()
                );
                $order_all[] = $sid;
            }

            $subject = $row['subject'];
            if ($subject != '' && $subject !== null) {
                // نخزن كل الأعمدة الخاصة بالدرجات لهذه المادة
                $students_all[$sid]['grades'][$subject] = array(
                    'month1'         => $row['month1'],
                    'month2'         => $row['month2'],
                    'month3'         => $row['month3'],
                    'daily1'         => $row['daily1'],
                    'semester1'      => $row['semester1'],
                    'midterm'        => $row['midterm'],
                    'month4'         => $row['month4'],
                    'month5'         => $row['month5'],
                    'month6'         => $row['month6'],
                    'daily2'         => $row['daily2'],
                    'semester2'      => $row['semester2'],
                    'annual_average' => $row['annual_average'],
                    'final_result'   => $row['final_result'],
                );

                if (!in_array($subject, $subjects_all, true)) {
                    $subjects_all[] = $subject;
                }
            }
        }
    }

    // ترتيب المواد
    if (!empty($subjects_all)) {
        $desired_order = array(
            'التربية الاسلامية',
            'اللغة العربية',
            'اللغة الانكليزية',
            'الرياضيات',
            'الكيمياء',
            'الاحياء',
            'الفيزياء',
            'الاجتماعيات',
            'الحاسوب',
            'اللغة الفرنسية',
            'اللغة الكردية'
        );

        $ordered = array();
        foreach ($desired_order as $subj) {
            if (in_array($subj, $subjects_all, true)) {
                $ordered[] = $subj;
            }
        }
        foreach ($subjects_all as $subj) {
            if (!in_array($subj, $ordered, true)) {
                $ordered[] = $subj;
            }
        }
        $subjects_all = $ordered;
    }

    // تحضير ملف الإكسل
    if (!empty($order_all)) {
        $filename_grade = $grade_type;
        $filename = "grades_all_stages_".$filename_grade.".xls";

        header("Content-Type: application/vnd.ms-excel; charset=utf-8");
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header("Pragma: no-cache");
        header("Expires: 0");

        echo "\xEF\xBB\xBF";

        echo "<table border='1'>";
        echo "<thead><tr>";
        echo "<th>ت</th>";
        echo "<th>المرحلة</th>";
        echo "<th>الشعبة</th>";
        echo "<th>اسم الطالب</th>";
        foreach ($subjects_all as $subj) {
            echo "<th>".esc($subj)."</th>";
        }
        echo "</tr></thead><tbody>";

        $i = 1;
        foreach ($order_all as $sid) {
            $st = $students_all[$sid];
            echo "<tr>";
            echo "<td>".$i++."</td>";
            echo "<td>".esc($st['stage'])."</td>";
            echo "<td>".esc($st['class'])."</td>";
            echo "<td>".esc($st['name'])."</td>";

            foreach ($subjects_all as $subj) {
                $gRow    = isset($st['grades'][$subj]) ? $st['grades'][$subj] : null;
                $display = '';

                if ($gRow !== null) {
                    $gradeNum = get_grade_value_from_row($gRow, $grade_type);
                    if ($gradeNum !== null) {
                        $display = rtrim(rtrim(number_format($gradeNum, 2, '.', ''), '0'), '.');
                    }
                }

                echo "<td>".esc($display)."</td>";
            }

            echo "</tr>";
        }

        echo "</tbody></table>";
    } else {
        header("Content-Type: text/plain; charset=utf-8");
        header("Content-Disposition: attachment; filename=\"grades_all_stages_empty.xls\"");
        echo "لا توجد بيانات درجات في نوع الدرجة المختار.";
    }
    exit;
}

// ------------------------------
// ٣) جلب قائمة المراحل (للواجهة)
// ------------------------------
$stages = array();
$resStages = mysqli_query($conn, "SELECT DISTINCT stage FROM students ORDER BY stage");
if ($resStages) {
    while ($row = mysqli_fetch_assoc($resStages)) {
        $stages[] = $row['stage'];
    }
}

// ------------------------------
// ٤) Pivot للمرحلة المختارة (عرض + تصدير مرحلي)
// ------------------------------
$students_data   = array(); // بيانات الطلاب مع الدرجات
$students_order  = array(); // للحفاظ على ترتيب العرض
$subjects        = array(); // قائمة المواد

if ($stage != '') {
    $stage_sql = mysqli_real_escape_string($conn, $stage);

    $where = "s.stage = '".$stage_sql."'";

    $sql = "
        SELECT 
            s.id   AS student_id,
            s.name AS student_name,
            s.class,
            g.subject,
            g.month1,
            g.month2,
            g.month3,
            g.daily1,
            g.semester1,
            g.midterm,
            g.month4,
            g.month5,
            g.month6,
            g.daily2,
            g.semester2,
            g.annual_average,
            g.final_result
        FROM students s
        LEFT JOIN grades g ON g.student_id = s.id
        WHERE ".$where."
        ORDER BY s.class, s.name
    ";

    $res = mysqli_query($conn, $sql);
    if ($res) {
        while ($row = mysqli_fetch_assoc($res)) {
            $sid = $row['student_id'];

            // إضافة الطالب إذا أول مرة نمر عليه
            if (!isset($students_data[$sid])) {
                $students_data[$sid] = array(
                    'name'   => $row['student_name'],
                    'class'  => $row['class'],
                    'grades' => array()
                );
                $students_order[] = $sid;
            }

            // مادة ودرجاتها
            $subject = $row['subject'];
            if ($subject != '' && $subject !== null) {
                $students_data[$sid]['grades'][$subject] = array(
                    'month1'         => $row['month1'],
                    'month2'         => $row['month2'],
                    'month3'         => $row['month3'],
                    'daily1'         => $row['daily1'],
                    'semester1'      => $row['semester1'],
                    'midterm'        => $row['midterm'],
                    'month4'         => $row['month4'],
                    'month5'         => $row['month5'],
                    'month6'         => $row['month6'],
                    'daily2'         => $row['daily2'],
                    'semester2'      => $row['semester2'],
                    'annual_average' => $row['annual_average'],
                    'final_result'   => $row['final_result'],
                );

                if (!in_array($subject, $subjects, true)) {
                    $subjects[] = $subject;
                }
            }
        }
    }

    // ✅ ترتيب المواد بالتسلسل المطلوب، وأي مادة أخرى تذهب في النهاية
    if (!empty($subjects)) {
        $desired_order = array(
            'التربية الاسلامية',
            'اللغة العربية',
            'اللغة الانكليزية',
            'الرياضيات',
            'الكيمياء',
            'الاحياء',
            'الفيزياء',
            'الاجتماعيات',
            'الحاسوب',
            'اللغة الفرنسية',
            'اللغة الكردية'
        );

        $ordered = array();

        foreach ($desired_order as $subj) {
            if (in_array($subj, $subjects, true)) {
                $ordered[] = $subj;
            }
        }
        foreach ($subjects as $subj) {
            if (!in_array($subj, $ordered, true)) {
                $ordered[] = $subj;
            }
        }

        $subjects = $ordered;
    }
}

// ------------------------------
// ٥) تصدير Excel للمرحلة الحالية فقط
// ------------------------------
if ($stage != '' && isset($_GET['export']) && $_GET['export'] === 'excel') {

    if (!empty($students_order)) {
        $filename_stage = preg_replace('/[^0-9\p{L}_-]+/u', '_', $stage);
        $filename_grade = $grade_type;
        $filename = "grades_matrix_".$filename_stage."_".$filename_grade.".xls";

        header("Content-Type: application/vnd.ms-excel; charset=utf-8");
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header("Pragma: no-cache");
        header("Expires: 0");

        echo "\xEF\xBB\xBF";

        echo "<table border='1'>";
        echo "<thead><tr>";
        echo "<th>ت</th>";
        echo "<th>اسم الطالب</th>";
        echo "<th>الشعبة</th>";
        foreach ($subjects as $subj) {
            echo "<th>".esc($subj)."</th>";
        }
        echo "<th>عدد مواد الإكمال</th>";
        echo "</tr></thead><tbody>";

        // مجاميع مواد الإكمال لكل مادة + عدد الطلبة المكملين ككل
        $subject_red_counts = array();
        foreach ($subjects as $subj) {
            $subject_red_counts[$subj] = 0;
        }
        $students_with_any_red = 0;

        $i = 1;
        foreach ($students_order as $sid) {
            $st = $students_data[$sid];
            echo "<tr>";
            echo "<td>".$i++."</td>";
            echo "<td>".esc($st['name'])."</td>";
            echo "<td>".esc($st['class'])."</td>";

            $redCount = 0;
            $hasRed   = false;

            foreach ($subjects as $subj) {
                $gRow    = isset($st['grades'][$subj]) ? $st['grades'][$subj] : null;
                $display = '';

                if ($gRow !== null) {
                    $gradeNum = get_grade_value_from_row($gRow, $grade_type);
                    if ($gradeNum !== null) {
                        if ($gradeNum < 50) {
                            $redCount++;
                            $hasRed = true;
                            $subject_red_counts[$subj]++;
                        }
                        $display = rtrim(rtrim(number_format($gradeNum, 2, '.', ''), '0'), '.');
                    }
                }

                echo "<td>".esc($display)."</td>";
            }

            if ($hasRed) {
                $students_with_any_red++;
            }

            echo "<td>".$redCount."</td>";
            echo "</tr>";
        }

        // ✅ صف المجاميع في نهاية الجدول
        echo "<tr>";
        echo "<td></td>";
        echo "<td>مجموع الطلبة المكملين</td>";
        echo "<td></td>";
        foreach ($subjects as $subj) {
            echo "<td>".$subject_red_counts[$subj]."</td>";
        }
        echo "<td>".$students_with_any_red."</td>";
        echo "</tr>";

        echo "</tbody></table>";
    } else {
        header("Content-Type: text/plain; charset=utf-8");
        header("Content-Disposition: attachment; filename=\"grades_matrix_empty.xls\"");
        echo "لا توجد بيانات درجات لهذه المرحلة في نوع الدرجة المختار.";
    }
    exit;
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
<meta charset="UTF-8">
<title>📊 كشف درجات الطلاب (مصفوفة مواد)</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;600;700&display=swap" rel="stylesheet">
<style>
*{box-sizing:border-box;margin:0;padding:0}
body{
    font-family:'Tajawal',sans-serif;
    background:radial-gradient(circle at 20% 20%,#1a1f3b,#0b0d18 80%);
    color:#fff;
    min-height:100vh;
    padding:20px 20px 10px 20px;
}

/* ✅ نفس الهيدر مع زر الرجوع الموجود في صفحة سجلات الدخول */
header{
    display:flex;
    justify-content:space-between;
    align-items:center;
    margin-bottom:15px;
}
header h1{
    font-size:22px;
}
header a{
    text-decoration:none;
    padding:7px 16px;
    border-radius:999px;
    background:linear-gradient(120deg,#00e6ff,#007bff);
    color:#fff;
    font-weight:600;
}

.page-title{
    text-align:center;
    margin-bottom:15px;
    font-size:24px;
    font-weight:700;
}
.filter-card{
    max-width:1100px;
    margin:0 auto 20px;
    background:#14172b;
    border-radius:14px;
    padding:15px 20px;
    box-shadow:0 10px 25px rgba(0,0,0,0.4);
}
.filter-row{
    display:flex;
    flex-wrap:wrap;
    gap:10px;
    align-items:flex-end;
}
.filter-group{
    flex:1 1 160px;
}
.filter-group label{
    display:block;
    margin-bottom:5px;
    font-size:13px;
}
.filter-group select{
    width:100%;
    padding:6px 8px;
    border-radius:8px;
    border:1px solid #444;
    background:#0f1120;
    color:#fff;
}
.btn{
    padding:8px 18px;
    border-radius:10px;
    border:none;
    cursor:pointer;
    font-weight:600;
    background:#2ecc71;
    color:#000;
}
.btn-print{
    background:#3498db;
    margin-right:10px;
}
.btn-excel{
    background:#1abc9c;
    color:#000;
    text-decoration:none;
    display:inline-block;
    margin-right:10px;
}
.btn-excel-all{
    background:#f39c12;
    color:#000;
    text-decoration:none;
    display:inline-block;
    margin-right:10px;
}

/* ✅ حاوية الجدول */
.table-wrapper{
    max-width:1100px;
    margin:0 auto 10px;
    background:#14172b;
    border-radius:14px;
    padding:10px;
    box-shadow:0 10px 25px rgba(0,0,0,0.4);
    overflow:auto;
    height:calc(100vh - 150px);
    position:relative;
    direction:ltr; /* شريط التمرير يمين */
}

/* الجدول نفسه RTL */
table{
    width:100%;
    border-collapse:separate;
    border-spacing:0;
    min-width:800px;
    direction:rtl;
}
th,td{
    border:1px solid #2b3050;
    padding:4px 6px;
    text-align:center;
    font-size:13px;
    white-space:nowrap;
    background:#14172b;
}

/* تثبيت صف العناوين */
thead th{
    background:#1f2440;
    position:sticky;
    top:0;
    z-index:4;
}

/* تدرج بسيط للصفوف على الشاشة */
tr:nth-child(even) td{
    background:#101326;
}
tr:hover td{
    background:#1a1f38;
}
.header-info{
    text-align:center;
    margin-bottom:8px;
    font-size:14px;
}

/* الشعبة كبادج */
td.class-cell span{
    display:inline-block;
    padding:2px 10px;
    border-radius:999px;
    font-size:12px;
    font-weight:600;
}
.sec-A{background:#1abc9c;color:#000;}
.sec-B{background:#3498db;color:#fff;}
.sec-C{background:#9b59b6;color:#fff;}
.sec-D{background:#e67e22;color:#000;}
.sec-E{background:#f1c40f;color:#000;}
.sec-F{background:#e74c3c;color:#fff;}

/* تلوين الدرجات حسب القيمة */
td.grade-cell{
    font-weight:600;
}
.grade-red{
    background:#c0392b !important;
    color:#fff !important;
}
.grade-yellow{
    background:#f1c40f !important;
    color:#000 !important;
}
.grade-blue{
    background:#2980b9 !important;
    color:#fff !important;
}
.grade-green{
    background:#27ae60 !important;
    color:#fff !important;
}

/* تنسيق اسم الطالب */
th.col-name,
td.col-name{
    text-align:right;
}

/* ✅ صف المجاميع في أسفل الجدول */
tr.summary-row td{
    background:#0b1220 !important;
    font-weight:700;
    border-top:2px solid #f59e0b;
}
.summary-cell{
    background:#0b1220 !important;
}

/* طباعة */
@media print{
    body{
        background:#fff;
        color:#000;
        padding:0;
    }
    .filter-card,
    .btn-print,
    .btn-excel,
    .btn-excel-all{
        display:none;
    }
    .table-wrapper{
        box-shadow:none;
        border:none;
        overflow:visible;
        height:auto;
        direction:rtl;
        background:#14172b;
        padding:0;
        margin:0 auto;
    }
    table{
        direction:rtl;
    }
    th,td{
        border:1px solid #000;
        font-size:13px;
    }

    /* عناوين المواد */
    thead th{
        color:#fff !important;
        font-weight:700 !important;
    }

    /* أسماء الطلبة */
    th.col-name,
    td.col-name{
        color:#fff !important;
        font-weight:700 !important;
    }

    /* سطر المرحلة – نوع الدرجة */
    .header-info,
    .header-info strong{
        color:#fff !important;
        font-weight:700;
    }

    /* تسلسل الطلبة (عمود ت) */
    thead th:first-child,
    tbody td:first-child{
        color:#fff !important;
        font-weight:700;
    }

    /* عمود عدد مواد الإكمال (آخر عمود) */
    tbody td:last-child{
        color:#fff !important;
        font-weight:700;
    }

    /* خلايا صف مجموع الطلبة المكملين */
    tr.summary-row td{
        color:#fff !important;
        font-weight:700;
    }
}
</style>
</head>
<body>

<!-- ✅ هيدر مع زر رجوع مثل صفحة سجلات الدخول -->
<header>
    <h1>📊 كشف درجات الطلاب (مصفوفة مواد)</h1>
    <a href="admin_dashboard.php">↩ العودة للوحة التحكم</a>
</header>

<div class="filter-card">
    <form method="get">
        <div class="filter-row">
            <div class="filter-group">
                <label>المرحلة</label>
                <select name="stage" required>
                    <option value="">اختر المرحلة</option>
                    <?php foreach ($stages as $st): ?>
                        <option value="<?php echo esc($st); ?>" <?php echo ($st == $stage ? 'selected' : ''); ?>>
                            <?php echo esc($st); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="filter-group">
                <label>نوع الدرجة</label>
                <select name="grade_type">
                    <?php foreach ($grade_types as $key => $label): ?>
                        <option value="<?php echo esc($key); ?>" <?php echo ($key == $grade_type ? 'selected' : ''); ?>>
                            <?php echo esc($label); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="filter-group" style="flex:0 0 auto;">
                <button type="submit" class="btn">عرض</button>
                <?php if ($stage != ''): ?>
                    <button type="button" class="btn btn-print" onclick="window.print();">طباعة</button>
                    <!-- تصدير المرحلة الحالية -->
                    <a href="?stage=<?php echo urlencode($stage); ?>&grade_type=<?php echo urlencode($grade_type); ?>&export=excel" class="btn-excel">
                        تصدير Excel (المرحلة الحالية)
                    </a>
                <?php endif; ?>
                <!-- ✅ تصدير كل المراحل (حسب نوع الدرجة المختار في الفورم) -->
                <button type="submit" name="export" value="excel_all" class="btn btn-excel-all">
                    تصدير Excel (كل المراحل)
                </button>
            </div>
        </div>
    </form>
</div>

<?php if ($stage == ''): ?>
    <div class="table-wrapper" style="text-align:center;padding:20px;">
        يرجى اختيار المرحلة ثم الضغط على زر <strong>عرض</strong>.
    </div>
<?php elseif (empty($students_order)): ?>
    <div class="table-wrapper" style="text-align:center;padding:20px;">
        لا توجد بيانات درجات لهذه المرحلة في نوع الدرجة المختار.
    </div>
<?php else: ?>
    <div class="table-wrapper">
        <div class="header-info">
            المرحلة: <strong><?php echo esc($stage); ?></strong>
            &nbsp;–&nbsp; نوع الدرجة: <strong><?php echo esc($grade_types[$grade_type]); ?></strong>
        </div>

        <table>
            <thead>
            <tr>
                <th>ت</th>
                <th class="col-name">اسم الطالب</th>
                <th>الشعبة</th>
                <?php foreach ($subjects as $subj): ?>
                    <th><?php echo esc($subj); ?></th>
                <?php endforeach; ?>
                <th>عدد مواد الإكمال</th>
            </tr>
            </thead>
            <tbody>
            <?php
            // مجاميع مواد الإكمال لكل مادة + عدد الطلبة المكملين
            $subject_red_counts = array();
            foreach ($subjects as $subj) {
                $subject_red_counts[$subj] = 0;
            }
            $students_with_any_red = 0;

            $i = 1;
            foreach ($students_order as $sid):
                $st = $students_data[$sid];

                // تجهيز لون الشعبة
                $className = $st['class'];
                $classCss  = '';
                switch ($className) {
                    case 'A': $classCss = 'sec-A'; break;
                    case 'B': $classCss = 'sec-B'; break;
                    case 'C': $classCss = 'sec-C'; break;
                    case 'D': $classCss = 'sec-D'; break;
                    case 'E': $classCss = 'sec-E'; break;
                    case 'F': $classCss = 'sec-F'; break;
                    default:  $classCss = '';     break;
                }

                $redCount = 0;
                $hasRed   = false;
            ?>
                <tr>
                    <td><?php echo $i++; ?></td>
                    <td class="col-name"><?php echo esc($st['name']); ?></td>
                    <td class="class-cell">
                        <?php if ($className != ''): ?>
                            <span class="<?php echo $classCss; ?>"><?php echo esc($className); ?></span>
                        <?php endif; ?>
                    </td>
                    <?php foreach ($subjects as $subj):
                        $gRow      = isset($st['grades'][$subj]) ? $st['grades'][$subj] : null;
                        $display   = '';
                        $cellClass = '';

                        if ($gRow !== null) {
                            $gradeNum = get_grade_value_from_row($gRow, $grade_type);
                            if ($gradeNum !== null) {

                                if ($gradeNum < 50) {
                                    $cellClass = 'grade-red';
                                    $redCount++;
                                    $hasRed = true;
                                    $subject_red_counts[$subj]++;
                                } elseif ($gradeNum < 60) {
                                    $cellClass = 'grade-yellow';
                                } elseif ($gradeNum < 80) {
                                    $cellClass = 'grade-blue';
                                } else {
                                    $cellClass = 'grade-green';
                                }

                                $display = rtrim(rtrim(number_format($gradeNum, 2, '.', ''), '0'), '.');
                            }
                        }
                    ?>
                        <td class="grade-cell <?php echo $cellClass; ?>"><?php echo esc($display); ?></td>
                    <?php endforeach; ?>
                    <td><?php echo $redCount; ?></td>
                </tr>
            <?php
                if ($hasRed) {
                    $students_with_any_red++;
                }
            endforeach;
            ?>

            <!-- ✅ صف المجاميع -->
            <tr class="summary-row">
                <td></td>
                <td class="col-name">مجموع الطلبة المكملين</td>
                <td></td>
                <?php foreach ($subjects as $subj): ?>
                    <td class="summary-cell"><?php echo (int)$subject_red_counts[$subj]; ?></td>
                <?php endforeach; ?>
                <td class="summary-cell"><?php echo (int)$students_with_any_red; ?></td>
            </tr>

            </tbody>
        </table>
    </div>
<?php endif; ?>

<script>
// 🔁 عند تحميل الصفحة، حرك شريط التمرير إلى أقصى اليمين
// حتى يظهر عمود "اسم الطالب" مباشرة
document.addEventListener('DOMContentLoaded', function () {
    var tw = document.querySelector('.table-wrapper');
    if (tw) {
        tw.scrollLeft = tw.scrollWidth;
    }
});
</script>

</body>
</html>
